<?php
	
class ET_RegistrationHelper {
	
	public const ERROR_WS_REGISTRATION = "ERROR_WS_REGISTRATION";
	public const ERROR_WS_EMAIL= "ERROR_WS_EMAIL";
	
	/***
	 * makes registrtaion of the new user
	 * 
	 * @param arr() $regData
	 * @throws Exception
	 * @return string - message for the UI
	 */
	public static function makeRegistration($regData) {
		
		if (!defined("VALIDATION_LINK_LIFETIME")) {
			$systemLogger->addSystemLogEntry(LOG_TYPE_ERROR, "VALIDATION_LINK_LIFETIME is not defined in ystem_defaults.php!");
			throw new Exception("Configureation error! VALIDATION_LINK_LIFETIME parameter is not defined in system_defaults.php!");
		}
		
		$msg 	= null;
		$result = null;
		
		$username 			= $regData["username"];
		$email 				= $regData["email"];
		$firstname 			= $regData["firstname"];
		$lastname 			= $regData["lastname"];
		$company 			= $regData["company"];
		$passwordHashed 	= password_hash($regData["pw"], PASSWORD_BCRYPT);
		$regDataForQuery	= [$username, 0, $passwordHashed, $email, $firstname, $lastname, $company];
		
		
		
		$dataSavingSuccess = ET_MetaDataConnector::getInstance()->saveRegistrationData($regDataForQuery);
		
		if ($dataSavingSuccess === false) {
			throw new Exception("DATA_SAVING_ERROR_DURING_REGISTRATION");
		} else {
			
			$systemLogger			= new ET_LoggingHelperSystem();
			$guid					= self::generateGUID();
			$verifyLink 			= self::generateVerifyLink($guid);
			$verifyDataSaveSuccess 	= ET_MetaDataConnector::getInstance()->saveVerifyData($regData, $guid);
			
			$regDataToResolve 						= [];
			$regDataToResolve["email"] 				= $regData["email"];
			$regDataToResolve["userName"] 			= $regData["username"];
			$regDataToResolve["verifylink"]			= $verifyLink;
			$regDataToResolve["linkLifetime"] 		= VALIDATION_LINK_LIFETIME;
			$regDataToResolve["companyName"] 		= $company;
			$regDataToResolve["langGrid"] 			= defined('GRID_AREA_LANGUAGE') ? GRID_AREA_LANGUAGE : 'en';
			
			$registrationWSResult 	= self::callRegistrationZagreusWS($regDataToResolve);
			
			//design changed: one Zagreus script makes regitstration also email sending
			//email error is handled there			
			if (!$registrationWSResult) {				
				$msg = self::ERROR_WS_REGISTRATION;
				$systemLogger->addSystemLogEntry(LOG_TYPE_ERROR, "Registration WS error for e-mail address: $email");
				
			} else {
				$msg = "REGISTRATION_WS_CALLS_SUCCESS";
			}
			return $msg;
		}
	}

	
	/***
	 * 
	 * Calls Zagreus WS to send https email to the customer with verification link.
	 * Variables to substitutein xml: $email, $verifylink, $username
	 * 
	 * @param unknown $regData
	 * @param unknown $verifyLink
	 * @throws Exception
	 * @return boolean
	 */
	public static function sendVerifyLinkViaZagreus ($regData, $verifyLink) {
		
		$systemLogger 		= new \ET_LoggingHelperSystem();
		
		if (!defined("VALIDATION_LINK_LIFETIME")) {
			$systemLogger->addSystemLogEntry(LOG_TYPE_ERROR, "VALIDATION_LINK_LIFETIME is not defined in ystem_defaults.php!");
			throw new Exception("Configureation error! VALIDATION_LINK_LIFETIME parameter is not defined in system_defaults.php!");
		}
		
		$email 				= $regData["email"];
		$username 			= $regData["username"];		
		$callSuccess 		= false;
		$dataToResolve		= [];
		$dataToResolve["email"] 		= $regData["email"];
		$dataToResolve["userName"] 		= $regData["username"];
		$dataToResolve["verifylink"]	= $verifyLink;			
		$dataToResolve["linkLifetime"] 	= VALIDATION_LINK_LIFETIME;
		$regDataToResolve["companyName"]= $regData["company"];
		$regDataToResolve["langGrid"] 	= defined('GRID_AREA_LANGUAGE') ? GRID_AREA_LANGUAGE : 'en';
		
		$zagreusConnection 	= ET_ConnHelper::getSystemZagreusConnection();
		
		if (defined("CUSTOM_CALLABLE_VERIFYEMAIL_WS")) {
			$callSuccess = ET_CallableObjectHelper::callWebServiceSimple($zagreusConnection, ET_ZAGREUS_CONNECTION, CUSTOM_CALLABLE_VERIFYEMAIL_WS, $dataToResolve);			
		} else {
			$systemLogger->addSystemLogEntry(LOG_TYPE_ERROR, "Cannot send verify e-mail! CUSTOM_CALLABLE_VERIFYEMAIL_WS is not defined in ystem_defaults.php!");
			throw new Exception("Configureation error! CUSTOM_CALLABLE_VERIFYEMAIL_WS parameter is not defined in system_defaults.php!");
		}
		return $callSuccess;
	}
	
	public static function callRegistrationZagreusWS ($dataToResolve) {
		
		$zagreusConnection 	= ET_ConnHelper::getSystemZagreusConnection();		
		$callSuccess 		= false;
		$systemLogger 		= new \ET_LoggingHelperSystem();
		
		if (defined("CUSTOM_CALLABLE_SELFSIGNIN_WS")) {
			$callSuccess = ET_CallableObjectHelper::callWebServiceSimple($zagreusConnection, ET_ZAGREUS_CONNECTION, CUSTOM_CALLABLE_SELFSIGNIN_WS, $dataToResolve);
		} else {
			$systemLogger->addSystemLogEntry(LOG_TYPE_ERROR, "Cannot start regiastration Zagreus script! CUSTOM_CALLABLE_SELFSIGNIN_WS is not dfeined in system_defaults.php");
			throw new Exception("Configureation error! CUSTOM_CALLABLE_SELFSIGNIN_WS parameter is not defined in system_defaults.php!");
		}
		
		return $callSuccess;
	}
	
	/**
	 * Generates a 8 characters long username
	 *
	 */
	public static function generateUsername ($userNameRe = "") {
		
		//8 characters long from e-mail hash -> upper case
		$username 				= substr(self::generateGUID(), 0, 8);		
		$usernameAlreadyExists = true;
		
		try {
			$usernameAlreadyExists 	= ET_MetaDataConnector::getInstance()->checkExistingUserName($username); 
		} catch (Exception $e) {
			return false;
		}
		
		if ($usernameAlreadyExists) {
			$username = self::generateUsername($username);
		} else {
			return $username;
		}
				
	}
	
	/**
	 * Genereate verify link
	 */
	public static function generateVerifyLink($guid) {
		
		//$guid = self::generateGUID();
		$host = isset($_SERVER['REMOTE_HOST']) ? $_SERVER['REMOTE_HOST'] : 
			(isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : $_SERVER['SERVER_ADDR']);
			
		if (isset($_SERVER['HTTP_X_FORWARDED_HOST']) && !is_null($_SERVER['HTTP_X_FORWARDED_HOST'])) {
			$host = $_SERVER['HTTP_X_FORWARDED_HOST'];
		}	
			
		$verifyHandlerPath 	= ET_PathHelper::getRelativePath("verify.php");
		// cut / from the end
		$verifyHandlerPath = substr($verifyHandlerPath, 0, strlen($verifyHandlerPath) - 1);
		$client				= isset($_SESSION[SESS_CLIENT]) ? $_SESSION[SESS_CLIENT] : CLIENT_SYSTEM;
		
		$link 				= "http://" . $host . $verifyHandlerPath . "?client=$client&vguid=" . $guid;
		$link = htmlentities($link);
		return $link;
	}
	
	
	public static function generateGUID () {
		return $username = sprintf('%04X%04X-%04X-%04X-%04X-%04X%04X%04X', mt_rand(0, 65535), mt_rand(0, 65535), mt_rand(0, 65535), mt_rand(16384, 20479), mt_rand(32768, 49151), mt_rand(0, 65535), mt_rand(0, 65535), mt_rand(0, 65535));
	}
	
	public static function activateUser($guid) {
		
		if (!defined("VALIDATION_LINK_LIFETIME")) {
			return ACTIVATION_CONFIG_ERROR;
		}
		
		$activationData = array();
		try {
			$activationData = ET_MetaDataConnector::getInstance()->getActivationData($guid);
		} catch (Exception $e) {
			return ACTIVATION_DB_ERROR;
		}
		
		$activationTS 	= time();
		$regTS			= $activationData['regTS'];
		$username 		= $activationData['username'];		
		$timeDiffSec	= $activationTS - $regTS;
		
		if (empty($activationData)) {
			return ACTIVATION_GUID_NOT_VALID;
		} else {
			if ($timeDiffSec / 3600 < VALIDATION_LINK_LIFETIME) {
				
				$activationResult = ET_MetaDataConnector::getInstance()->activateUser($username);
				return $activationResult;
				
			} else {
				return ACTIVATION_TIMEOUT;
			}
		}
		
		
	}
	
	public static function checkPwPolicy($pw) {
		
		try {
			$pwChecker = new ET_PwCheck($pw);
			return $pwChecker->checkPW();
		} catch (Exception $e) {
			$logger = new ET_LoggingHelperSystem();
			$logger->addSystemLogEntry(LOG_TYPE_ERROR, "Confioguraition error! Password policy is not defined correctly in system_defaults.php! Registration does not work!");
			throw new Exception($e->getMessage());
		}
		
	}
	
	public static function resendVerificationEmail ($email, $pw, $company) {
		
		$userInfo 		= null;
		$ret 			= [];
		$ret["error"] 	= false;
		$ret["message"] = "";
		
		try {
			$userInfo =  ET_MetaDataConnector::getInstance()->getUserInfoByEmailAndPw($email, $pw);
			
			if (is_array($userInfo)) {
				$success 				= false;
				$regData 				= [];				
				$regData["username"] 	= $userInfo["username"];
				$verificationGUID		= $userInfo["verify_guid"];
				$regData["email"] 		= $email;
				$regData["company"] 	= $company;
				$verifyLink 			= self::generateVerifyLink($verificationGUID);
				
				try {					
					$success 	= self::sendVerifyLinkViaZagreus($regData, $verifyLink);
				} catch (Exception $e) {					
					$ret["error"] 	= true;
					$ret["message"] = $e->getMessage();
				}
				
				if ($success) {					
					$ret["error"] 	= false;
					$ret["message"] = ET_DisplayHelper::localize("RESEND_EMAIL_VIA_ZAGREUS_SUCCESS", array(), PAGE_TYPE_GRID, true);
				} else {					
					$ret["error"] 	= true;
					$ret["message"] = ET_DisplayHelper::localize("RESEND_EMAIL_VIA_ZAGREUS_FAILED", array(), PAGE_TYPE_GRID, true);
				}				
			} else {				
				$ret["error"] 	= true;
				$ret["message"] = ET_DisplayHelper::localize("RESEND_VERIFY_DATA_RETRIEVE_ERROR", array(), PAGE_TYPE_GRID, true);
				$ret["message"] = $userInfo == "VERIFY_RESEND_AUTH_ERROR" ? ET_DisplayHelper::localize("VERIFY_RESEND_AUTH_ERROR", array(), PAGE_TYPE_GRID, true) :
									$ret["message"];
			}
			return $ret;
		} catch (Exception $e) {
			echo $e->getMessage();
			throw $e;
		}
		
		
	}
	
}